// Settings page functionality

async function showSettingsView() {
    const user = getCurrentUser();
    if (!user) {
        showView('auth');
        return;
    }
    
    showView('settings');
    await loadSettingsData();
}

async function loadSettingsData() {
    const user = getCurrentUser();
    if (!user) return;
    
    // Set user info
    const initial = (user.firstName || user.email || 'U').charAt(0).toUpperCase();
    document.getElementById('settings-user-initial').textContent = initial;
    document.getElementById('settings-user-name').textContent = user.firstName 
        ? `${user.firstName} ${user.lastName || ''}`.trim() 
        : 'User';
    document.getElementById('settings-user-email').textContent = user.email;
    
    // Load subscription status
    try {
        const subscription = await api.getSubscriptionStatus();
        
        const planBadge = subscription.plan === 'pro_monthly' 
            ? '<span style="background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: white; padding: 2px 8px; border-radius: 12px; font-size: 10px; font-weight: 600;">POWER BUYER</span>'
            : '<span style="background: #e0e0e0; color: #666; padding: 2px 8px; border-radius: 12px; font-size: 10px; font-weight: 600;">FREE</span>';
        
        document.getElementById('settings-user-plan').innerHTML = planBadge;
        
        // Show subscription details
        const detailsContainer = document.getElementById('subscription-details');
        const actionsContainer = document.getElementById('subscription-actions');
        
        if (subscription.plan === 'pro_monthly') {
            let statusText = 'Active';
            let statusColor = '#4CAF50';
            
            if (subscription.cancelAtPeriodEnd) {
                statusText = 'Canceling';
                statusColor = '#FF9800';
            }
            
            detailsContainer.innerHTML = `
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="color: #666;">Plan</span>
                    <span style="font-weight: 600; color: #1565C0;">Power Buyer</span>
                </div>
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="color: #666;">Price</span>
                    <span style="font-weight: 600;">$12.99/month</span>
                </div>
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="color: #666;">Status</span>
                    <span style="font-weight: 600; color: ${statusColor};">${statusText}</span>
                </div>
                ${subscription.currentPeriodEnd ? `
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <span style="color: #666;">${subscription.cancelAtPeriodEnd ? 'Access Until' : 'Next Billing'}</span>
                    <span style="font-weight: 600;">${new Date(subscription.currentPeriodEnd).toLocaleDateString()}</span>
                </div>
                ` : ''}
                ${subscription.cancelAtPeriodEnd ? `
                <div style="background: #fff3e0; padding: 10px; border-radius: 6px; margin-top: 12px; font-size: 12px; color: #E65100;">
                    <i class="fas fa-info-circle"></i> Your subscription will remain active until the end of your billing period, then automatically cancel.
                </div>
                ` : ''}
            `;
            
            actionsContainer.innerHTML = `
                <button id="manage-subscription-btn" class="btn btn-secondary" style="width: 100%; margin-bottom: 10px;">
                    <i class="fas fa-external-link-alt"></i> Manage Subscription
                </button>
                <p style="font-size: 11px; color: #999; text-align: center; margin: 0;">
                    Update payment method, view invoices, or cancel subscription
                </p>
            `;
            
            document.getElementById('manage-subscription-btn').addEventListener('click', openBillingPortal);
        } else {
            detailsContainer.innerHTML = `
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="color: #666;">Plan</span>
                    <span style="font-weight: 600;">Free (Starter)</span>
                </div>
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="color: #666;">Daily Scans</span>
                    <span style="font-weight: 600;">${subscription.features?.scansPerDay || 10}/day</span>
                </div>
                <div style="background: #e3f2fd; padding: 10px; border-radius: 6px; margin-top: 12px; font-size: 12px; color: #1565C0;">
                    <i class="fas fa-star"></i> Upgrade to Power Buyer for unlimited scans, AI cost estimates, and ad-free experience!
                </div>
            `;
            
            actionsContainer.innerHTML = `
                <button id="upgrade-btn" class="btn btn-search" style="width: 100%; background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%);">
                    <i class="fas fa-crown"></i> Upgrade to Power Buyer
                </button>
            `;
            
            document.getElementById('upgrade-btn').addEventListener('click', () => {
                showView('pricing');
            });
        }
    } catch (error) {
        console.error('Failed to load subscription:', error);
        document.getElementById('subscription-details').innerHTML = `
            <p style="color: #666; font-size: 13px;">Unable to load subscription details</p>
        `;
    }
}

async function openBillingPortal() {
    const btn = document.getElementById('manage-subscription-btn');
    if (!btn) return;
    
    const originalText = btn.innerHTML;
    
    try {
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Opening...';
        
        const result = await api.createPortalSession();
        
        if (result.url) {
            window.open(result.url, '_blank');
        } else {
            alert('Unable to open billing portal. Please try again.');
        }
    } catch (error) {
        console.error('Failed to open billing portal:', error);
        alert('Failed to open billing portal: ' + (error.message || 'Unknown error'));
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
}

async function handlePasswordChange() {
    const currentPassword = document.getElementById('current-password').value;
    const newPassword = document.getElementById('new-password').value;
    const confirmPassword = document.getElementById('confirm-new-password').value;
    
    const errorDiv = document.getElementById('password-change-error');
    const successDiv = document.getElementById('password-change-success');
    
    errorDiv.style.display = 'none';
    successDiv.style.display = 'none';
    
    if (!currentPassword || !newPassword || !confirmPassword) {
        errorDiv.textContent = 'Please fill in all password fields';
        errorDiv.style.display = 'block';
        return;
    }
    
    if (newPassword.length < 6) {
        errorDiv.textContent = 'New password must be at least 6 characters';
        errorDiv.style.display = 'block';
        return;
    }
    
    if (newPassword !== confirmPassword) {
        errorDiv.textContent = 'New passwords do not match';
        errorDiv.style.display = 'block';
        return;
    }
    
    const btn = document.getElementById('change-password-btn');
    const originalText = btn.innerHTML;
    
    try {
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Updating...';
        
        const result = await api.changePassword(currentPassword, newPassword);
        
        if (result.success) {
            successDiv.textContent = 'Password changed successfully!';
            successDiv.style.display = 'block';
            
            // Clear the form
            document.getElementById('current-password').value = '';
            document.getElementById('new-password').value = '';
            document.getElementById('confirm-new-password').value = '';
        } else {
            errorDiv.textContent = result.error || 'Failed to change password';
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        console.error('Change password error:', error);
        errorDiv.textContent = error.error || error.message || 'Failed to change password. Please check your current password.';
        errorDiv.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
}

// Initialize settings functionality
document.addEventListener('DOMContentLoaded', () => {
    const backBtn = document.getElementById('back-to-main-from-settings');
    if (backBtn) {
        backBtn.addEventListener('click', () => showView('main'));
    }
    
    const changePasswordBtn = document.getElementById('change-password-btn');
    if (changePasswordBtn) {
        changePasswordBtn.addEventListener('click', handlePasswordChange);
    }
    
    const logoutBtn = document.getElementById('settings-logout-btn');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', async () => {
            if (confirm('Are you sure you want to logout?')) {
                await api.logout();
                showView('main');
                if (typeof updateAuthUI === 'function') {
                    updateAuthUI();
                }
            }
        });
    }
});
