let currentSubscription = null;
let subscriptionPollingInterval = null;

async function loadPricingPage() {
  // Always load plans first (doesn't require auth)
  try {
    const plans = await api.getPlans();
    updatePriceIds(plans);
  } catch (error) {
    console.error('Error loading plans:', error);
  }
  
  // Then try to load subscription status (requires auth)
  try {
    const subscription = await api.getSubscriptionStatus();
    currentSubscription = subscription;
    updateCurrentPlanBanner(subscription);
    updatePlanBadges(subscription.plan);
  } catch (error) {
    // User not logged in - show default state
    updateCurrentPlanBanner({ plan: 'free', features: { scansPerDay: 10 } });
    updatePlanBadges('free');
  }
}

function updateCurrentPlanBanner(subscription) {
  const banner = document.getElementById('current-plan-banner');
  const planName = document.getElementById('current-plan-name');
  const planScans = document.getElementById('current-plan-scans');

  if (!subscription || subscription.plan === 'free') {
    planName.textContent = 'Starter (Free)';
    const limit = subscription?.features?.scansPerDay || 10;
    planScans.textContent = `${limit} scans per day`;
  } else if (subscription.plan === 'pro_monthly') {
    planName.textContent = 'Power Buyer';
    planScans.textContent = 'Unlimited scans • $12.99/month';
  }

  banner.style.display = 'block';
}

function updatePlanBadges(currentPlan) {
  document.getElementById('free-plan-badge').style.display = currentPlan === 'free' ? 'block' : 'none';
  document.getElementById('pro-monthly-badge').style.display = currentPlan === 'pro_monthly' ? 'block' : 'none';

  document.querySelectorAll('.pricing-select-btn').forEach(btn => {
    const plan = btn.dataset.plan;
    if (plan === currentPlan) {
      btn.textContent = 'Current Plan';
      btn.disabled = true;
      btn.style.background = '#f5f5f5';
      btn.style.color = '#666';
      btn.style.cursor = 'not-allowed';
    } else if (plan === 'free') {
      btn.textContent = 'Downgrade to Free';
      btn.disabled = false;
      btn.style.cursor = 'pointer';
    } else if (plan === 'pro_monthly' && currentPlan !== 'pro_monthly') {
      btn.textContent = currentPlan === 'free' ? 'Subscribe Now' : 'Switch Plan';
      btn.disabled = false;
    }
  });
}

function updatePriceIds(plans) {
  const monthlyBtn = document.querySelector('[data-plan="pro_monthly"]');

  if (monthlyBtn && plans.plans?.pro_monthly?.stripePriceId) {
    monthlyBtn.dataset.priceId = plans.plans.pro_monthly.stripePriceId;
  }
}

document.addEventListener('DOMContentLoaded', () => {
  // Load price IDs on page load
  loadPricingPage();
  
  document.querySelectorAll('.pricing-select-btn').forEach(btn => {
    btn.addEventListener('click', async () => {
      const plan = btn.dataset.plan;

      if (plan === 'free') {
        showManageSubscriptionPortal();
        return;
      }

      try {
        btn.disabled = true;
        btn.textContent = 'Loading...';

        // Always fetch fresh price ID from API to ensure it's available
        const plans = await api.getPlans();
        const priceId = plans.plans?.pro_monthly?.stripePriceId;
        
        if (!priceId) {
          throw new Error('Price ID not available from server');
        }
        
        console.log('Using price ID:', priceId);
        const { url } = await api.createCheckoutSession(priceId);
        
        if (url) {
          window.open(url, '_blank');
          startSubscriptionPolling(btn);
        } else {
          throw new Error('No checkout URL returned');
        }
      } catch (error) {
        console.error('Error creating checkout session:', error);
        alert('Failed to start checkout: ' + error.message);
        btn.disabled = false;
        btn.textContent = 'Subscribe Now';
      }
    });
  });

  const backBtn = document.getElementById('back-to-main-from-pricing');
  if (backBtn) {
    backBtn.addEventListener('click', () => {
      stopSubscriptionPolling();
      showMainView();
    });
  }
});

async function showManageSubscriptionPortal() {
  try {
    const { url } = await api.createPortalSession();
    window.open(url, '_blank');
  } catch (error) {
    console.error('Error opening portal:', error);
    alert('Failed to open subscription management. Please try again.');
  }
}

function showPricingView() {
  hideAllViews();
  document.getElementById('pricing-view').style.display = 'block';
  loadPricingPage();
}

function startSubscriptionPolling(btn) {
  if (subscriptionPollingInterval) {
    clearInterval(subscriptionPollingInterval);
  }
  
  btn.textContent = 'Waiting for payment...';
  
  let pollCount = 0;
  const maxPolls = 60;
  
  subscriptionPollingInterval = setInterval(async () => {
    pollCount++;
    
    if (pollCount > maxPolls) {
      clearInterval(subscriptionPollingInterval);
      subscriptionPollingInterval = null;
      btn.disabled = false;
      btn.textContent = 'Subscribe Now';
      return;
    }
    
    try {
      if (window.yopickySubscriptionCache) {
        const token = localStorage.getItem('yopicky_token');
        if (token) {
          const payload = JSON.parse(atob(token.split('.')[1]));
          const cacheKey = `sub_${payload.userId}`;
          delete window.yopickySubscriptionCache[cacheKey];
        }
      }
      
      const subscription = await api.getSubscriptionStatus();
      
      if (subscription && subscription.plan === 'pro_monthly') {
        clearInterval(subscriptionPollingInterval);
        subscriptionPollingInterval = null;
        
        currentSubscription = subscription;
        updateCurrentPlanBanner(subscription);
        updatePlanBadges(subscription.plan);
        
        if (typeof updateAuthUI === 'function') {
          updateAuthUI();
        }
        
        if (!window.upgradeSuccessShown) {
          window.upgradeSuccessShown = true;
          showUpgradeSuccessMessage();
        }
        
        return;
      }
    } catch (error) {
    }
  }, 5000);
}

function stopSubscriptionPolling() {
  if (subscriptionPollingInterval) {
    clearInterval(subscriptionPollingInterval);
    subscriptionPollingInterval = null;
  }
}

function showUpgradeSuccessMessage() {
  const banner = document.getElementById('current-plan-banner');
  if (banner) {
    const successDiv = document.createElement('div');
    successDiv.id = 'upgrade-success-message';
    successDiv.style.cssText = 'background: linear-gradient(135deg, #4CAF50, #45a049); color: white; padding: 12px; border-radius: 8px; margin-bottom: 16px; text-align: center; font-weight: 500;';
    successDiv.innerHTML = '🎉 Welcome to Power Buyer! Your subscription is now active.';
    banner.parentNode.insertBefore(successDiv, banner);
    
    setTimeout(() => {
      successDiv.remove();
    }, 10000);
  }
}
