function extractListingId() {
    const url = window.location.href;
    let listingId = null;
    let auctionSite = null;
    let country = 'US';

    // IAAI - Insurance Auto Auctions
    // URL formats:
    // - /VehicleDetail/43668469~US or /VehicleDetail/43668469-US
    // - /vehicledetails/41506227 (lowercase)
    // - /Vehicle/43668469~US
    // - ?itemNumber=12345678 or ?stock=12345678
    if (url.includes('iaai.com')) {
        auctionSite = 'IAAI';
        // Try VehicleDetail/vehicledetails/Vehicle URL formats (most common)
        const vehicleMatch = url.match(/(?:VehicleDetail|vehicledetails|Vehicle)\/(\d+)/i);
        if (vehicleMatch) {
            listingId = `IAAI_${vehicleMatch[1]}`;
        } else {
            // Try itemNumber query/path format
            const itemMatch = url.match(/itemNumber[=\/](\d+)/i);
            if (itemMatch) {
                listingId = `IAAI_${itemMatch[1]}`;
            } else {
                // Try stock query/path format
                const stockMatch = url.match(/stock[=\/](\d+)/i);
                if (stockMatch) {
                    listingId = `IAAI_${stockMatch[1]}`;
                } else {
                    // Fallback: try to find 8-digit number in URL
                    const numMatch = url.match(/\/(\d{7,8})(?:[~\-\/\?]|$)/);
                    if (numMatch) listingId = `IAAI_${numMatch[1]}`;
                }
            }
        }
    }
    
    // Copart
    // URL formats:
    // - /lot/36508943 or /lot/36508943/
    // - ?lotNumber=36508943
    else if (url.includes('copart.com')) {
        auctionSite = 'Copart';
        // Try /lot/ path format (most common)
        const lotMatch = url.match(/\/lot\/(\d+)/i);
        if (lotMatch) {
            listingId = `COPART_${lotMatch[1]}`;
        } else {
            // Try lotNumber query parameter
            const queryMatch = url.match(/lotNumber[=](\d+)/i);
            if (queryMatch) {
                listingId = `COPART_${queryMatch[1]}`;
            } else {
                // Fallback: try DOM element
                const lotEl = document.querySelector('[data-lot], .lot-number, [data-lotNumber]');
                if (lotEl) {
                    const lotNum = lotEl.getAttribute('data-lot') || 
                                   lotEl.getAttribute('data-lotnumber') || 
                                   lotEl.textContent.match(/\d{7,8}/)?.[0];
                    if (lotNum) listingId = `COPART_${lotNum}`;
                }
            }
        }
    }
    
    // SCA Auction (both scaauction.com and sca.auction domains)
    // Real URL formats from sca.auction:
    // - /en/1012659592-2012-volvo-s80 (ID-YEAR-MAKE-MODEL format)
    // - /en/1022657892-2006-honda-element
    // - /vehicle/12345678
    // - ?lotId=12345678
    else if (url.includes('scaauction.com') || url.includes('sca.auction')) {
        auctionSite = 'SCA Auction';
        // Primary format: /en/{10-digit-ID}-{year}-{make}-{model}
        const scaMainMatch = url.match(/\/(?:en\/)?(\d{9,11})-\d{4}-/i);
        if (scaMainMatch) {
            listingId = `SCA_${scaMainMatch[1]}`;
        } else {
            // Alternative: /en/{ID} without model info
            const scaIdMatch = url.match(/\/(?:en\/)?(\d{9,11})(?:[\/\?-]|$)/i);
            if (scaIdMatch) {
                listingId = `SCA_${scaIdMatch[1]}`;
            } else {
                // Try /vehicle/ path format
                const vehicleMatch = url.match(/\/vehicle\/(\d+)/i);
                if (vehicleMatch) {
                    listingId = `SCA_${vehicleMatch[1]}`;
                } else {
                    // Try lotId query parameter
                    const lotMatch = url.match(/lotId[=](\d+)/i);
                    if (lotMatch) {
                        listingId = `SCA_${lotMatch[1]}`;
                    }
                }
            }
        }
    }
    
    // eRepairables
    // URL formats:
    // - /salvage-cars-auction/.../12345678
    // - /vehicle/id/12345678
    // - ?auctionId=12345678
    else if (url.includes('erepairables.com')) {
        auctionSite = 'eRepairables';
        // Try vehicle/id path format
        const vehicleIdMatch = url.match(/vehicle\/id\/(\d+)/i);
        if (vehicleIdMatch) {
            listingId = `EREPAIR_${vehicleIdMatch[1]}`;
        } else {
            // Try auctionId query parameter
            const auctionMatch = url.match(/auctionId[=](\d+)/i);
            if (auctionMatch) {
                listingId = `EREPAIR_${auctionMatch[1]}`;
            } else {
                // Try salvage path with ID
                const salvageMatch = url.match(/salvage[^\/]*\/(\d{5,10})/i);
                if (salvageMatch) {
                    listingId = `EREPAIR_${salvageMatch[1]}`;
                } else {
                    // Fallback: look for any large number that could be an ID
                    const numMatch = url.match(/\/(\d{6,10})(?:[\/\?]|$)/);
                    if (numMatch) listingId = `EREPAIR_${numMatch[1]}`;
                }
            }
        }
    }
    
    // AutoBidMaster
    // Real URL formats from autobidmaster.com:
    // - /en/search/lot/93274065/copart-2014-toyota-corolla-l-... (most common)
    // - /en/lot/12345678/...
    // - /search/lot/12345678/...
    // - ?lot=12345678
    else if (url.includes('autobidmaster.com')) {
        auctionSite = 'AutoBidMaster';
        // Primary format: /search/lot/{ID}/ or /lot/{ID}/
        const lotMatch = url.match(/\/(?:search\/)?lot\/(\d+)/i);
        if (lotMatch) {
            listingId = `AUTOBID_${lotMatch[1]}`;
        } else {
            // Try lot query parameter
            const queryMatch = url.match(/[\?&]lot[=](\d+)/i);
            if (queryMatch) {
                listingId = `AUTOBID_${queryMatch[1]}`;
            } else {
                // Fallback: look for 8-digit number in URL path
                const numMatch = url.match(/\/(\d{8})(?:[\/\?]|$)/);
                if (numMatch) listingId = `AUTOBID_${numMatch[1]}`;
            }
        }
    }

    // Detect country from URL
    if (url.includes('.ca/') || url.includes('/ca/') || url.includes('canada')) {
        country = 'CA';
    } else if (url.includes('.co.uk/') || url.includes('/uk/') || url.includes('uk.')) {
        country = 'GB';
    } else if (url.includes('.com.au/') || url.includes('/au/') || url.includes('australia')) {
        country = 'AU';
    } else if (url.includes('.de/') || url.includes('/de/') || url.includes('germany')) {
        country = 'DE';
    }

    return { listingId, auctionSite, country };
}

chrome.runtime.onMessage.addListener(function(cmd, sender, sendResponse) {
    switch(cmd) {
    case "getHtml":
        const listingData = extractListingId();
        sendResponse({
            title: document.title, 
            url: window.location.href, 
            html: document.documentElement.innerHTML,
            ...listingData
        });
        break;
    case "getHeadTitle":
        sendResponse(document.getElementsByTagName("title")[0].innerHTML);
        break;
    case "getListingId":
        sendResponse(extractListingId());
        break;
    default:
        sendResponse(null);
    }
});