// Collections management functionality

let currentPartToSave = null;
let selectedCollectionId = null;
let currentCollectionForEstimate = null;

// Show collections view
async function showCollectionsView() {
    const user = getCurrentUser();
    if (!user) {
        alert('Please login to view collections');
        showView('auth');
        return;
    }
    
    // Clear search context when viewing collections
    window.currentSearchContext = null;
    showView('collections');
    if (typeof updateAuthUI === 'function') {
        updateAuthUI();
    }
    await loadCollections();
}

// Load and display collections
async function loadCollections() {
    const container = document.getElementById('collections-list');
    
    try {
        const collections = await getUserCollections();
        
        if (collections.length === 0) {
            container.innerHTML = `
                <div class="category-box design-1">
                    <h4 class="car-name" style="text-align: center; color: #666;">
                        No collections yet. Save parts to start organizing!
                    </h4>
                </div>
            `;
            return;
        }
        
        const html = await Promise.all(collections.map(async (collection) => {
            const parts = await getCollectionParts(collection.id);
            
            return `
                <div class="category-box design-1" style="margin-bottom: 20px;">
                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 12px;">
                        <div style="flex: 1;">
                            <h4 class="car-name" style="margin: 0 0 4px 0;">${collection.name}</h4>
                            ${collection.description ? `<p style="margin: 0; font-size: 12px; color: #666;">${collection.description}</p>` : ''}
                            <p style="margin: 4px 0 0 0; font-size: 12px; color: #999;">${parts.length} parts</p>
                        </div>
                        <div style="display: flex; gap: 8px;">
                            <button class="view-collection-btn" data-collection-id="${collection.id}" style="background: #2196F3; color: white; border: none; padding: 6px 12px; border-radius: 6px; cursor: pointer; font-size: 12px;">
                                <i class="fas fa-eye"></i> View
                            </button>
                            <button class="delete-collection-btn" data-collection-id="${collection.id}" style="background: #ff4444; color: white; border: none; padding: 6px 12px; border-radius: 6px; cursor: pointer; font-size: 12px;">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `;
        }));
        
        container.innerHTML = html.join('');
        
        document.querySelectorAll('.view-collection-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const collectionId = e.currentTarget.getAttribute('data-collection-id');
                await viewCollectionParts(collectionId);
            });
        });
        
        document.querySelectorAll('.delete-collection-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const collectionId = e.currentTarget.getAttribute('data-collection-id');
                if (confirm('Are you sure you want to delete this collection and all its parts?')) {
                    try {
                        await deleteCollection(collectionId);
                        await loadCollections();
                    } catch (error) {
                        alert('Failed to delete collection: ' + error.message);
                    }
                }
            });
        });
        
    } catch (error) {
        container.innerHTML = `
            <div class="category-box design-1">
                <h4 class="car-name error">
                    Failed to load collections: ${error.message}
                </h4>
            </div>
        `;
    }
}

// View parts in a collection
async function viewCollectionParts(collectionId) {
    const container = document.getElementById('collections-list');
    currentCollectionForEstimate = collectionId;
    
    try {
        const parts = await getCollectionParts(collectionId);
        const collections = await getUserCollections();
        const collection = collections.find(c => c.id === collectionId);
        
        if (!collection) {
            alert('Collection not found');
            return;
        }
        
        // Check for cached AI estimate
        let cachedEstimate = null;
        try {
            const cachedResult = await api.getCachedCollectionEstimate(collectionId);
            if (cachedResult.hasEstimate) {
                cachedEstimate = cachedResult.estimate;
            }
        } catch (err) {
            console.log('No cached estimate found');
        }
        
        // Calculate summary totals
        let totalPartsPrice = 0;
        let totalShippingCost = 0;
        
        parts.forEach(part => {
            const price = parseFloat((part.price || '0').toString().replace(/[^0-9.]/g, '')) || 0;
            const shipping = parseFloat((part.shippingCost || '0').toString().replace(/[^0-9.]/g, '')) || 0;
            totalPartsPrice += price;
            totalShippingCost += shipping;
        });
        
        const grandTotal = totalPartsPrice + totalShippingCost;
        
        if (parts.length === 0) {
            container.innerHTML = `
                <div style="margin-bottom: 20px;">
                    <button id="back-to-collections-list" class="submit-btn">
                        <i class="fas fa-arrow-left"></i> Back to Collections
                    </button>
                </div>
                <h3 style="margin-bottom: 20px;">${collection.name}</h3>
                <div class="category-box design-1">
                    <h4 class="car-name" style="text-align: center; color: #666;">
                        No parts in this collection yet
                    </h4>
                </div>
            `;
            
            document.getElementById('back-to-collections-list').addEventListener('click', loadCollections);
            return;
        }
        
        // Collection summary box - show cached results or button
        let summaryBox;
        
        if (cachedEstimate) {
            // Show cached AI results
            const profitColor = cachedEstimate.potentialProfit >= 0 ? '#4CAF50' : '#f44336';
            const profitIcon = cachedEstimate.potentialProfit >= 0 ? 'fa-arrow-up' : 'fa-arrow-down';
            
            let breakdownHtml = '';
            if (cachedEstimate.carBreakdowns && cachedEstimate.carBreakdowns.length > 1) {
                breakdownHtml = `
                    <h4 style="margin: 16px 0 12px 0; color: #1565C0;"><i class="fas fa-car"></i> Per-Vehicle Breakdown</h4>
                    ${cachedEstimate.carBreakdowns.map(car => `
                        <div style="background: #f5f5f5; padding: 12px; border-radius: 8px; margin-bottom: 10px;">
                            <div style="font-weight: 600; margin-bottom: 8px;">${car.year} ${car.make} ${car.model}</div>
                            <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 8px; font-size: 12px;">
                                <div>Parts (${car.partCount}): <strong>$${car.totalParts.toFixed(2)}</strong></div>
                                <div>Shipping: <strong>$${car.totalShipping.toFixed(2)}</strong></div>
                                <div>Subtotal: <strong>$${car.subtotal.toFixed(2)}</strong></div>
                                <div>Market Value: <strong style="color: #2196F3;">$${car.marketValue.toFixed(2)}</strong></div>
                            </div>
                        </div>
                    `).join('')}
                `;
            }
            
            summaryBox = `
                <div style="background: linear-gradient(135deg, #e8f5e9 0%, #c8e6c9 100%); padding: 16px; border-radius: 10px; margin-bottom: 20px; border: 1px solid #81C784;">
                    <h4 style="margin: 0 0 12px 0; color: #2E7D32;"><i class="fas fa-check-circle"></i> AI Cost Analysis Complete</h4>
                    
                    <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px; margin-bottom: 12px;">
                        <div style="background: white; padding: 10px; border-radius: 6px; text-align: center;">
                            <div style="font-size: 11px; color: #666;">Total Parts Cost</div>
                            <div style="font-size: 16px; font-weight: 600; color: #2196F3;">$${cachedEstimate.totalParts.toFixed(2)}</div>
                        </div>
                        <div style="background: white; padding: 10px; border-radius: 6px; text-align: center;">
                            <div style="font-size: 11px; color: #666;">Total Shipping</div>
                            <div style="font-size: 16px; font-weight: 600; color: #4CAF50;">$${cachedEstimate.totalShipping.toFixed(2)}</div>
                        </div>
                    </div>
                    
                    <div style="background: white; padding: 12px; border-radius: 6px; text-align: center; margin-bottom: 12px;">
                        <div style="font-size: 11px; color: #666;">Grand Total Investment</div>
                        <div style="font-size: 20px; font-weight: 700; color: #1565C0;">$${cachedEstimate.grandTotal.toFixed(2)}</div>
                    </div>
                    
                    <div style="background: white; padding: 12px; border-radius: 6px; text-align: center; margin-bottom: 12px;">
                        <div style="font-size: 11px; color: #666;">Estimated Market Value</div>
                        <div style="font-size: 20px; font-weight: 700; color: #2E7D32;">$${cachedEstimate.marketValue.toFixed(2)}</div>
                        <div style="font-size: 10px; color: #999; margin-top: 4px;">${cachedEstimate.marketValueNote || ''}</div>
                    </div>
                    
                    <div style="background: white; padding: 12px; border-radius: 6px; text-align: center;">
                        <div style="font-size: 11px; color: #666;">Potential ${cachedEstimate.potentialProfit >= 0 ? 'Profit' : 'Loss'}</div>
                        <div style="font-size: 20px; font-weight: 700; color: ${profitColor};">
                            <i class="fas ${profitIcon}"></i> $${Math.abs(cachedEstimate.potentialProfit).toFixed(2)}
                            <span style="font-size: 12px;">(${cachedEstimate.profitPercentage}%)</span>
                        </div>
                    </div>
                    
                    ${breakdownHtml}
                </div>
            `;
        } else {
            // Show basic summary with AI button
            summaryBox = `
                <div style="background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%); padding: 16px; border-radius: 10px; margin-bottom: 20px;">
                    <h4 style="margin: 0 0 12px 0; color: #1565C0;"><i class="fas fa-calculator"></i> Collection Summary</h4>
                    <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px; margin-bottom: 12px;">
                        <div style="background: white; padding: 10px; border-radius: 6px; text-align: center;">
                            <div style="font-size: 11px; color: #666;">Parts Cost</div>
                            <div style="font-size: 16px; font-weight: 600; color: #2196F3;">$${totalPartsPrice.toFixed(2)}</div>
                        </div>
                        <div style="background: white; padding: 10px; border-radius: 6px; text-align: center;">
                            <div style="font-size: 11px; color: #666;">Shipping</div>
                            <div style="font-size: 16px; font-weight: 600; color: #4CAF50;">$${totalShippingCost.toFixed(2)}</div>
                        </div>
                    </div>
                    <div style="background: white; padding: 12px; border-radius: 6px; text-align: center; margin-bottom: 12px;">
                        <div style="font-size: 11px; color: #666;">Grand Total</div>
                        <div style="font-size: 20px; font-weight: 700; color: #1565C0;">$${grandTotal.toFixed(2)}</div>
                    </div>
                    <button id="ai-cost-estimator-btn" data-collection-id="${collectionId}" style="width: 100%; background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: white; border: none; padding: 12px; border-radius: 8px; cursor: pointer; font-size: 14px; font-weight: 600;">
                        <i class="fas fa-robot"></i> AI Cost Estimator - Get Market Value
                    </button>
                </div>
            `;
        }
        
        const partsHtml = parts.map(part => {
            const priceVal = parseFloat((part.price || '0').toString().replace(/[^0-9.]/g, '')) || 0;
            const shippingVal = parseFloat((part.shippingCost || '0').toString().replace(/[^0-9.]/g, '')) || 0;
            
            return `
            <div class="category-box design-1" style="margin-bottom: 12px;">
                <div style="display: flex; justify-content: space-between; align-items: start;">
                    <div style="flex: 1;">
                        <h4 class="car-name" style="margin: 0 0 4px 0; font-size: 13px;">${part.partName || 'Unnamed Part'}</h4>
                        <div style="display: flex; gap: 12px; flex-wrap: wrap; margin-top: 6px;">
                            ${priceVal > 0 ? `<span style="font-weight: bold; color: #2196F3;">$${priceVal.toFixed(2)}</span>` : ''}
                            ${shippingVal > 0 ? `<span style="font-size: 11px; color: #4CAF50;"><i class="fas fa-truck"></i> +$${shippingVal.toFixed(2)}</span>` : '<span style="font-size: 11px; color: #4CAF50;"><i class="fas fa-truck"></i> Free</span>'}
                            ${part.condition ? `<span style="font-size: 11px; color: #666;">${part.condition}</span>` : ''}
                        </div>
                        ${part.carYear && part.carMake ? `<div style="font-size: 10px; color: #999; margin-top: 4px;">${part.carYear} ${part.carMake} ${part.carModel || ''}</div>` : ''}
                    </div>
                    <div style="display: flex; gap: 6px;">
                        ${part.url ? `<a href="${part.url}" target="_blank" class="submit-btn" style="padding: 6px 10px; font-size: 11px; text-decoration: none;"><i class="fas fa-external-link-alt"></i></a>` : ''}
                        <button class="remove-part-btn" data-collection-id="${collectionId}" data-part-id="${part.id}" style="background: #ff4444; color: white; border: none; padding: 6px 10px; border-radius: 6px; cursor: pointer; font-size: 11px;">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
            </div>
        `}).join('');
        
        container.innerHTML = `
            <div style="margin-bottom: 20px;">
                <button id="back-to-collections-list" class="submit-btn">
                    <i class="fas fa-arrow-left"></i> Back to Collections
                </button>
            </div>
            <h3 style="margin-bottom: 16px;">${collection.name}</h3>
            ${summaryBox}
            <div id="ai-estimate-results" style="display: none;"></div>
            <h4 style="margin-bottom: 12px; color: #666;">Parts (${parts.length})</h4>
            ${partsHtml}
        `;
        
        document.getElementById('back-to-collections-list').addEventListener('click', loadCollections);
        
        // AI Cost Estimator button handler (only exists when no cached estimate)
        const aiBtn = document.getElementById('ai-cost-estimator-btn');
        if (aiBtn) {
            aiBtn.addEventListener('click', async function() {
                await runAICostEstimator(collectionId);
            });
        }
        
        document.querySelectorAll('.remove-part-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                const partId = e.currentTarget.getAttribute('data-part-id');
                const collId = e.currentTarget.getAttribute('data-collection-id');
                
                if (confirm('Remove this part from the collection?')) {
                    try {
                        await removePartFromCollection(collId, partId);
                        await viewCollectionParts(collId);
                    } catch (error) {
                        alert('Failed to remove part: ' + error.message);
                    }
                }
            });
        });
        
    } catch (error) {
        alert('Failed to load collection parts: ' + error.message);
    }
}

// Run AI Cost Estimator
async function runAICostEstimator(collectionId, includeBreakdowns = false) {
    const resultsContainer = document.getElementById('ai-estimate-results');
    const btn = document.getElementById('ai-cost-estimator-btn');
    
    try {
        btn.disabled = true;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Analyzing...';
        
        const result = await getCollectionCostEstimate(collectionId, includeBreakdowns);
        
        if (result.upgradeRequired) {
            resultsContainer.innerHTML = `
                <div style="background: #fff3e0; padding: 16px; border-radius: 10px; margin-bottom: 20px; border: 1px solid #FFB74D;">
                    <h4 style="margin: 0 0 8px 0; color: #E65100;"><i class="fas fa-lock"></i> Power Buyer Feature</h4>
                    <p style="margin: 0 0 12px 0; color: #666; font-size: 13px;">AI Cost Estimator with market value research requires a Power Buyer subscription.</p>
                    <button onclick="showPricingView()" style="background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: white; border: none; padding: 10px 20px; border-radius: 6px; cursor: pointer; font-weight: 600;">
                        <i class="fas fa-crown"></i> Upgrade Now
                    </button>
                </div>
            `;
            resultsContainer.style.display = 'block';
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-robot"></i> AI Cost Estimator - Get Market Value';
            return;
        }
        
        // Check for mixed vehicles
        if (result.isMixedVehicles && !includeBreakdowns) {
            resultsContainer.innerHTML = `
                <div style="background: #fff3e0; padding: 16px; border-radius: 10px; margin-bottom: 20px; border: 1px solid #FFB74D;">
                    <h4 style="margin: 0 0 8px 0; color: #E65100;"><i class="fas fa-exclamation-triangle"></i> Mixed Vehicle Collection</h4>
                    <p style="margin: 0 0 12px 0; color: #666; font-size: 13px;">
                        This collection contains parts from <strong>${result.vehicleCount} different vehicles</strong>. 
                        For the most accurate cost estimate, we recommend saving parts from the same car (make/model) in separate collections.
                    </p>
                    <div style="display: flex; gap: 10px;">
                        <button id="cancel-mixed-estimate" style="background: #f5f5f5; color: #666; border: 1px solid #ddd; padding: 10px 20px; border-radius: 6px; cursor: pointer;">
                            Cancel
                        </button>
                        <button id="continue-mixed-estimate" style="background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%); color: white; border: none; padding: 10px 20px; border-radius: 6px; cursor: pointer; font-weight: 600;">
                            Continue with Breakdown
                        </button>
                    </div>
                </div>
            `;
            resultsContainer.style.display = 'block';
            
            document.getElementById('cancel-mixed-estimate').addEventListener('click', () => {
                resultsContainer.style.display = 'none';
            });
            
            document.getElementById('continue-mixed-estimate').addEventListener('click', async () => {
                await runAICostEstimator(collectionId, true);
            });
            
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-robot"></i> AI Cost Estimator - Get Market Value';
            return;
        }
        
        // Display results
        const profitColor = result.potentialProfit >= 0 ? '#4CAF50' : '#f44336';
        const profitIcon = result.potentialProfit >= 0 ? 'fa-arrow-up' : 'fa-arrow-down';
        
        let breakdownHtml = '';
        if (result.carBreakdowns && result.carBreakdowns.length > 1) {
            breakdownHtml = `
                <h4 style="margin: 16px 0 12px 0; color: #1565C0;"><i class="fas fa-car"></i> Per-Vehicle Breakdown</h4>
                ${result.carBreakdowns.map(car => `
                    <div style="background: #f5f5f5; padding: 12px; border-radius: 8px; margin-bottom: 10px;">
                        <div style="font-weight: 600; margin-bottom: 8px;">${car.year} ${car.make} ${car.model}</div>
                        <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 8px; font-size: 12px;">
                            <div>Parts (${car.partCount}): <strong>$${car.totalParts.toFixed(2)}</strong></div>
                            <div>Shipping: <strong>$${car.totalShipping.toFixed(2)}</strong></div>
                            <div>Subtotal: <strong>$${car.subtotal.toFixed(2)}</strong></div>
                            <div>Market Value: <strong style="color: #2196F3;">$${car.marketValue.toFixed(2)}</strong></div>
                        </div>
                    </div>
                `).join('')}
            `;
        }
        
        // Reload the view to show cached results (this will hide the button and show the results)
        await viewCollectionParts(collectionId);
        
    } catch (error) {
        if (resultsContainer) {
            resultsContainer.innerHTML = `
                <div style="background: #ffebee; padding: 16px; border-radius: 10px; margin-bottom: 20px; border: 1px solid #ef9a9a;">
                    <h4 style="margin: 0 0 8px 0; color: #c62828;"><i class="fas fa-exclamation-circle"></i> Error</h4>
                    <p style="margin: 0; color: #666; font-size: 13px;">${error.message || 'Failed to generate cost estimate. Please try again.'}</p>
                </div>
            `;
            resultsContainer.style.display = 'block';
        }
        if (btn) {
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-robot"></i> AI Cost Estimator - Get Market Value';
        }
    }
}

// Show save to collection modal
async function showSaveToCollectionModal(partData) {
    const user = getCurrentUser();
    if (!user) {
        alert('Please login to save parts to collections');
        showView('auth');
        return;
    }
    
    currentPartToSave = partData;
    selectedCollectionId = null;
    
    const modal = document.getElementById('save-to-collection-modal');
    const modalListContainer = document.getElementById('modal-collections-list');
    const errorDiv = document.getElementById('save-collection-error');
    
    errorDiv.style.display = 'none';
    document.getElementById('new-collection-name').value = '';
    document.getElementById('new-collection-description').value = '';
    
    try {
        const collections = await getUserCollections();
        
        if (collections.length === 0) {
            modalListContainer.innerHTML = '<p style="color: #999; font-size: 13px; text-align: center; padding: 20px;">No collections yet. Create one below!</p>';
        } else {
            modalListContainer.innerHTML = collections.map(c => `
                <div class="collection-option" data-collection-id="${c.id}" style="padding: 10px; margin-bottom: 8px; border: 2px solid #ddd; border-radius: 6px; cursor: pointer; transition: all 0.2s;">
                    <div style="font-weight: 600; margin-bottom: 2px;">${c.name}</div>
                    ${c.description ? `<div style="font-size: 12px; color: #666;">${c.description}</div>` : ''}
                </div>
            `).join('');
            
            document.querySelectorAll('.collection-option').forEach(option => {
                option.addEventListener('click', async function() {
                    document.querySelectorAll('.collection-option').forEach(o => {
                        o.style.borderColor = '#ddd';
                        o.style.background = 'transparent';
                    });
                    
                    this.style.borderColor = '#2196F3';
                    this.style.background = '#e3f2fd';
                    selectedCollectionId = this.getAttribute('data-collection-id');
                    
                    setTimeout(async () => {
                        try {
                            await addPartToCollection(selectedCollectionId, currentPartToSave);
                            alert('Part saved to collection!');
                            hideSaveToCollectionModal();
                        } catch (error) {
                            const errorDiv = document.getElementById('save-collection-error');
                            errorDiv.textContent = 'Failed to save part: ' + error.message;
                            errorDiv.style.display = 'block';
                        }
                    }, 300);
                });
            });
        }
        
        modal.style.display = 'flex';
    } catch (error) {
        alert('Failed to load collections: ' + error.message);
    }
}

// Hide save to collection modal
function hideSaveToCollectionModal() {
    document.getElementById('save-to-collection-modal').style.display = 'none';
    currentPartToSave = null;
    selectedCollectionId = null;
}

// Handle save to collection
async function handleSaveToCollection() {
    const newCollectionName = document.getElementById('new-collection-name').value.trim();
    const newCollectionDescription = document.getElementById('new-collection-description').value.trim();
    const errorDiv = document.getElementById('save-collection-error');
    
    try {
        let collectionId = selectedCollectionId;
        
        if (newCollectionName) {
            const newCollection = await createCollection(newCollectionName, newCollectionDescription);
            collectionId = newCollection.collection.id;
        } else if (!selectedCollectionId) {
            errorDiv.textContent = 'Please select a collection or create a new one';
            errorDiv.style.display = 'block';
            return;
        }
        
        await addPartToCollection(collectionId, currentPartToSave);
        
        alert('Part saved to collection!');
        hideSaveToCollectionModal();
    } catch (error) {
        errorDiv.textContent = 'Failed to save part: ' + error.message;
        errorDiv.style.display = 'block';
    }
}

// Initialize collections functionality
document.addEventListener('DOMContentLoaded', () => {
    const backBtn = document.getElementById('back-to-main-from-collections');
    if (backBtn) {
        backBtn.addEventListener('click', () => showView('main'));
    }
    
    const createBtn = document.getElementById('create-collection-btn');
    if (createBtn) {
        createBtn.addEventListener('click', async () => {
            const name = prompt('Enter collection name:');
            if (name) {
                const description = prompt('Enter description (optional):') || '';
                try {
                    await createCollection(name, description);
                    await loadCollections();
                } catch (error) {
                    alert('Failed to create collection: ' + error.message);
                }
            }
        });
    }
    
    const confirmBtn = document.getElementById('confirm-save-to-collection');
    if (confirmBtn) {
        confirmBtn.addEventListener('click', handleSaveToCollection);
    }
    
    const cancelBtn = document.getElementById('cancel-save-to-collection');
    if (cancelBtn) {
        cancelBtn.addEventListener('click', hideSaveToCollectionModal);
    }
});
