// Update auth UI based on login status
async function updateAuthUI() {
    const user = getCurrentUser();
    const authButtonsContainer = document.getElementById('auth-buttons');
    const userGreeting = document.getElementById('user-greeting');
    
    if (user) {
        // Check subscription status
        let isPowerBuyer = false;
        try {
            const subscription = await api.getSubscriptionStatus();
            isPowerBuyer = subscription && subscription.plan === 'pro_monthly';
        } catch (error) {
        }
        
        // User is logged in - build navigation buttons including Settings
        authButtonsContainer.innerHTML = `
            ${isPowerBuyer ? `
                <span id="power-buyer-badge" style="background: linear-gradient(135deg, #FFD700, #FFA500); color: #333; padding: 4px 8px; border-radius: 12px; font-size: 10px; font-weight: 600; white-space: nowrap;">
                    POWER BUYER
                </span>
            ` : `
                <button class="submit-btn upgrade-btn" id="upgrade-btn" style="font-size: 11px; padding: 8px 12px; background: linear-gradient(135deg, #FF9800 0%, #F57C00 100%);">
                    <i class="fas fa-crown"></i> Upgrade
                </button>
            `}
            <button class="submit-btn" id="my-collections-btn" style="font-size: 11px; padding: 8px 12px;">
                <i class="fas fa-folder"></i> Collections
            </button>
            <button class="submit-btn" id="settings-nav-btn" style="font-size: 11px; padding: 8px 12px;">
                <i class="fas fa-cog"></i> Settings
            </button>
            <button class="submit-btn logout-btn" id="logout-btn" style="font-size: 11px; padding: 8px 12px;">
                <i class="fas fa-sign-out-alt"></i> Logout
            </button>
        `;
        
        // Show user greeting
        if (userGreeting && user.firstName) {
            userGreeting.textContent = `Hi ${user.firstName}`;
            userGreeting.style.display = 'block';
        }
        
        // Add event listeners
        if (!isPowerBuyer) {
            document.getElementById('upgrade-btn').addEventListener('click', showPricingView);
        }
        document.getElementById('my-collections-btn').addEventListener('click', showCollectionsView);
        document.getElementById('settings-nav-btn').addEventListener('click', showSettingsView);
        document.getElementById('logout-btn').addEventListener('click', handleLogout);
    } else {
        // Hide settings button
        const settingsBtn = document.getElementById('settings-btn');
        if (settingsBtn) {
            settingsBtn.style.display = 'none';
        }
        
        // User is not logged in
        authButtonsContainer.innerHTML = `
            <button class="submit-btn" id="login-link-btn" style="font-size: 11px; padding: 8px 12px;">
                <i class="fas fa-sign-in-alt"></i> Login
            </button>
        `;
        
        // Hide user greeting
        if (userGreeting) {
            userGreeting.style.display = 'none';
        }
        
        // Add event listener
        document.getElementById('login-link-btn').addEventListener('click', () => showView('auth'));
    }
}

// Handle login
async function handleLogin(event) {
    event.preventDefault();
    
    const email = document.getElementById('login-email').value.trim();
    const password = document.getElementById('login-password').value;
    const errorDiv = document.getElementById('login-error');
    
    if (!email || !password) {
        errorDiv.textContent = 'Please enter email and password';
        errorDiv.style.display = 'block';
        return;
    }
    
    try {
        await login(email, password);
        errorDiv.style.display = 'none';
        updateAuthUI();
        
        // Ensure settings button is visible after login
        const settingsBtn = document.getElementById('settings-btn');
        if (settingsBtn) {
            settingsBtn.style.display = 'inline-flex';
        }
        
        showView('main');
        
        // Re-extract car data from current page after login
        setTimeout(() => {
            onloadEvent(0);
        }, 100);
        
        // Clear form
        document.getElementById('login-email').value = '';
        document.getElementById('login-password').value = '';
    } catch (error) {
        let errorMessage = 'Login failed. Please try again.';
        
        if (error.message.includes('connect to server')) {
            errorMessage = 'Unable to connect to server. Please check your internet connection.';
        } else if (error.message.includes('verify your email')) {
            errorMessage = 'Your email is not verified yet. We will resend a verification code to your email.';
            
            try {
                await resendOTP(email);
                showOTPForm(email);
                return;
            } catch (resendError) {
                errorMessage = 'Please verify your email. Failed to send verification code.';
            }
        } else if (error.message.includes('Invalid credentials') || error.message.includes('User not found')) {
            errorMessage = 'Invalid email or password. Please check your credentials.';
        } else if (error.message) {
            errorMessage = error.message;
        }
        
        errorDiv.textContent = errorMessage;
        errorDiv.style.display = 'block';
    }
}

// Handle registration
async function handleRegister(event) {
    event.preventDefault();
    
    const email = document.getElementById('register-email').value.trim();
    const password = document.getElementById('register-password').value;
    const firstName = document.getElementById('register-firstname').value.trim();
    const lastName = document.getElementById('register-lastname').value.trim();
    const errorDiv = document.getElementById('register-error');
    
    if (!email || !password) {
        errorDiv.textContent = 'Please enter email and password';
        errorDiv.style.display = 'block';
        return;
    }
    
    try {
        const data = await register(email, password, firstName, lastName);
        errorDiv.style.display = 'none';
        
        // Show OTP verification form
        showOTPForm(email);
        
        // Clear registration form
        document.getElementById('register-email').value = '';
        document.getElementById('register-password').value = '';
        document.getElementById('register-firstname').value = '';
        document.getElementById('register-lastname').value = '';
    } catch (error) {
        let errorMessage = 'Registration failed. Please try again.';
        
        if (error.message.includes('connect to server')) {
            errorMessage = 'Unable to connect to server. Please check your internet connection.';
        } else if (error.message.includes('already exists') || error.message.includes('email is already registered')) {
            errorMessage = 'This email is already registered. Please login instead.';
        } else if (error.message.includes('invalid email')) {
            errorMessage = 'Please enter a valid email address.';
        } else if (error.message.includes('password')) {
            errorMessage = 'Password must be at least 6 characters long.';
        } else if (error.message) {
            errorMessage = error.message;
        }
        
        errorDiv.textContent = errorMessage;
        errorDiv.style.display = 'block';
    }
}

// Show OTP verification form
function showOTPForm(email) {
    document.getElementById('login-form').style.display = 'none';
    document.getElementById('register-form').style.display = 'none';
    document.getElementById('otp-form').style.display = 'block';
    document.getElementById('otp-email-display').textContent = email;
    document.getElementById('otp-code').value = '';
    
    // Store email for OTP verification
    localStorage.setItem('pending_verification_email', email);
}

// Handle OTP verification
async function handleVerifyOTP(event) {
    event.preventDefault();
    
    const otp = document.getElementById('otp-code').value.trim();
    const email = localStorage.getItem('pending_verification_email');
    const errorDiv = document.getElementById('otp-error');
    
    if (!otp) {
        errorDiv.textContent = 'Please enter the verification code';
        errorDiv.style.display = 'block';
        return;
    }
    
    if (otp.length !== 6) {
        errorDiv.textContent = 'Verification code must be 6 digits';
        errorDiv.style.display = 'block';
        return;
    }
    
    try {
        await verifyOTP(email, otp);
        errorDiv.style.display = 'none';
        localStorage.removeItem('pending_verification_email');
        updateAuthUI();
        showView('main');
        document.getElementById('otp-code').value = '';
    } catch (error) {
        let errorMessage = 'Verification failed. Please try again.';
        
        if (error.message.includes('Invalid or expired')) {
            errorMessage = 'Invalid or expired code. Please check the code and try again.';
        } else if (error.message.includes('connect to server')) {
            errorMessage = 'Unable to connect to server. Please check your internet connection.';
        } else if (error.message) {
            errorMessage = error.message;
        }
        
        errorDiv.textContent = errorMessage;
        errorDiv.style.display = 'block';
    }
}

// Handle resend OTP
async function handleResendOTP(event) {
    event.preventDefault();
    
    const email = localStorage.getItem('pending_verification_email');
    const errorDiv = document.getElementById('otp-error');
    
    try {
        await resendOTP(email);
        errorDiv.style.display = 'none';
        
        const successDiv = document.createElement('div');
        successDiv.style.color = '#4CAF50';
        successDiv.style.marginBottom = '12px';
        successDiv.textContent = 'Verification code sent! Please check your email.';
        document.getElementById('otp-form').insertBefore(successDiv, document.getElementById('otp-error'));
        
        setTimeout(() => {
            successDiv.remove();
        }, 3000);
    } catch (error) {
        errorDiv.textContent = 'Failed to resend code. Please try again.';
        errorDiv.style.display = 'block';
    }
}

// Handle logout
async function handleLogout() {
    await logout();
    updateAuthUI();
    showView('main');
}

// Handle forgot password request
async function handleForgotPassword(event) {
    event.preventDefault();
    
    const email = document.getElementById('forgot-email').value.trim();
    const errorDiv = document.getElementById('forgot-error');
    
    if (!email) {
        errorDiv.textContent = 'Please enter your email';
        errorDiv.style.display = 'block';
        return;
    }
    
    try {
        const result = await forgotPassword(email);
        
        if (result.success) {
            // Store email for reset flow
            window.resetPasswordEmail = email;
            
            // Show reset password form
            document.getElementById('forgot-password-form').style.display = 'none';
            document.getElementById('reset-password-form').style.display = 'block';
            document.getElementById('reset-email-display').textContent = email;
        } else {
            errorDiv.textContent = result.error || 'Failed to send reset code';
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        errorDiv.textContent = 'An error occurred. Please try again.';
        errorDiv.style.display = 'block';
    }
}

// Handle password reset
async function handleResetPassword(event) {
    event.preventDefault();
    
    const code = document.getElementById('reset-code').value.trim();
    const newPassword = document.getElementById('new-password').value;
    const confirmPassword = document.getElementById('confirm-password').value;
    const errorDiv = document.getElementById('reset-error');
    
    if (!code || !newPassword || !confirmPassword) {
        errorDiv.textContent = 'Please fill in all fields';
        errorDiv.style.display = 'block';
        return;
    }
    
    if (newPassword !== confirmPassword) {
        errorDiv.textContent = 'Passwords do not match';
        errorDiv.style.display = 'block';
        return;
    }
    
    if (newPassword.length < 6) {
        errorDiv.textContent = 'Password must be at least 6 characters';
        errorDiv.style.display = 'block';
        return;
    }
    
    try {
        const result = await resetPassword(window.resetPasswordEmail, code, newPassword);
        
        if (result.success) {
            alert('Password reset successfully! Please login with your new password.');
            
            // Clear forms and show login
            document.getElementById('reset-code').value = '';
            document.getElementById('new-password').value = '';
            document.getElementById('confirm-password').value = '';
            document.getElementById('reset-password-form').style.display = 'none';
            document.getElementById('login-form').style.display = 'block';
        } else {
            errorDiv.textContent = result.error || 'Failed to reset password';
            errorDiv.style.display = 'block';
        }
    } catch (error) {
        errorDiv.textContent = 'An error occurred. Please try again.';
        errorDiv.style.display = 'block';
    }
}

// Handle resend reset code
async function handleResendResetCode(event) {
    event.preventDefault();
    
    try {
        const result = await forgotPassword(window.resetPasswordEmail);
        
        if (result.success) {
            alert('New reset code sent to your email');
        }
    } catch (error) {
        alert('Failed to resend code. Please try again.');
    }
}

// Initialize auth UI on load
document.addEventListener('DOMContentLoaded', () => {
    // Update UI based on current auth status
    updateAuthUI();
    
    // Auth form event listeners
    document.getElementById('login-btn').addEventListener('click', handleLogin);
    document.getElementById('register-btn').addEventListener('click', handleRegister);
    document.getElementById('verify-otp-btn').addEventListener('click', handleVerifyOTP);
    document.getElementById('resend-otp-btn').addEventListener('click', handleResendOTP);
    
    // Form toggle listeners
    document.getElementById('show-register').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('login-form').style.display = 'none';
        document.getElementById('register-form').style.display = 'block';
        document.getElementById('otp-form').style.display = 'none';
        document.getElementById('forgot-password-form').style.display = 'none';
        document.getElementById('reset-password-form').style.display = 'none';
    });
    
    document.getElementById('show-login').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('register-form').style.display = 'none';
        document.getElementById('login-form').style.display = 'block';
        document.getElementById('otp-form').style.display = 'none';
        document.getElementById('forgot-password-form').style.display = 'none';
        document.getElementById('reset-password-form').style.display = 'none';
    });
    
    document.getElementById('forgot-password-link').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('login-form').style.display = 'none';
        document.getElementById('register-form').style.display = 'none';
        document.getElementById('otp-form').style.display = 'none';
        document.getElementById('reset-password-form').style.display = 'none';
        document.getElementById('forgot-password-form').style.display = 'block';
    });
    
    document.getElementById('back-to-login-from-otp').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('otp-form').style.display = 'none';
        document.getElementById('login-form').style.display = 'block';
        localStorage.removeItem('pending_verification_email');
    });
    
    document.getElementById('back-to-login-from-forgot').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('forgot-password-form').style.display = 'none';
        document.getElementById('login-form').style.display = 'block';
    });
    
    document.getElementById('back-to-login-from-reset').addEventListener('click', (e) => {
        e.preventDefault();
        document.getElementById('reset-password-form').style.display = 'none';
        document.getElementById('login-form').style.display = 'block';
    });
    
    // Password reset listeners
    document.getElementById('send-reset-code-btn').addEventListener('click', handleForgotPassword);
    document.getElementById('reset-password-btn').addEventListener('click', handleResetPassword);
    document.getElementById('resend-reset-code-btn').addEventListener('click', handleResendResetCode);
    
    // Back to main listeners
    document.getElementById('back-to-main-from-login').addEventListener('click', (e) => {
        e.preventDefault();
        showView('main');
    });
    
    document.getElementById('back-to-main-from-register').addEventListener('click', (e) => {
        e.preventDefault();
        showView('main');
    });
    
    document.getElementById('back-to-main-btn').addEventListener('click', () => {
        showView('main');
    });
    
    // Verify token on load
    verifyToken().then(user => {
        if (user) {
            updateAuthUI();
        }
    });
});
