// API configuration - Development server URL
const API_BASE_URL = 'https://ee1c092d-3167-4f59-b56c-638708490601-00-32pbmujpx1fit.janeway.replit.dev/api';

// View management functions (loaded early so all scripts can use them)
function hideAllViews() {
    document.getElementById('main-view').style.display = 'none';
    document.getElementById('auth-view').style.display = 'none';
    document.getElementById('collections-view').style.display = 'none';
    document.getElementById('pricing-view').style.display = 'none';
    const settingsView = document.getElementById('settings-view');
    if (settingsView) settingsView.style.display = 'none';
}

function showView(viewName) {
    hideAllViews();
    document.getElementById(`${viewName}-view`).style.display = 'block';
}

function showMainView() {
    showView('main');
}

// Get the auth token from localStorage
function getAuthToken() {
    return localStorage.getItem('yopicky_auth_token');
}

// Set the auth token in localStorage
function setAuthToken(token) {
    localStorage.setItem('yopicky_auth_token', token);
}

// Remove the auth token from localStorage
function clearAuthToken() {
    localStorage.removeItem('yopicky_auth_token');
    localStorage.removeItem('yopicky_user');
}

// Get the current user from localStorage
function getCurrentUser() {
    const userJson = localStorage.getItem('yopicky_user');
    return userJson ? JSON.parse(userJson) : null;
}

// Set the current user in localStorage
function setCurrentUser(user) {
    localStorage.setItem('yopicky_user', JSON.stringify(user));
}

// API request wrapper
async function apiRequest(endpoint, options = {}) {
    const token = getAuthToken();
    const headers = {
        'Content-Type': 'application/json',
        ...options.headers,
    };

    if (token) {
        headers['Authorization'] = `Bearer ${token}`;
    }

    try {
        const response = await fetch(`${API_BASE_URL}${endpoint}`, {
            ...options,
            headers,
        });

        const data = await response.json();

        if (!response.ok) {
            const error = new Error(data.error || 'Request failed');
            error.upgradeRequired = data.upgradeRequired || false;
            error.error = data.error;
            error.message = data.error || data.message || 'Request failed';
            throw error;
        }

        return data;
    } catch (error) {
        if (error.message === 'Failed to fetch') {
            throw new Error('Unable to connect to server. Please check your internet connection.');
        }
        throw error;
    }
}

// Auth API functions
async function register(email, password, firstName, lastName) {
    const data = await apiRequest('/auth/register', {
        method: 'POST',
        body: JSON.stringify({ email, password, firstName, lastName }),
    });

    // Registration now returns only a message (OTP sent) - no token yet
    return data;
}

async function login(email, password) {
    const data = await apiRequest('/auth/login', {
        method: 'POST',
        body: JSON.stringify({ email, password }),
    });

    setAuthToken(data.token);
    setCurrentUser(data.user);
    
    try {
        const subscription = await getSubscriptionStatus();
        if (!window.yopickySubscriptionCache) {
            window.yopickySubscriptionCache = {};
        }
        window.yopickySubscriptionCache[`sub_${data.user.id}`] = {
            data: subscription,
            timestamp: Date.now()
        };
    } catch (error) {
        // Silently ignore - subscription will be fetched when needed
    }
    
    return data;
}

async function logout() {
    clearAuthToken();
}

async function verifyToken() {
    try {
        const data = await apiRequest('/auth/verify');
        setCurrentUser(data.user);
        return data.user;
    } catch (error) {
        clearAuthToken();
        return null;
    }
}

// Saved searches API functions
async function saveSearch(searchData) {
    return await apiRequest('/searches/save', {
        method: 'POST',
        body: JSON.stringify(searchData),
    });
}

async function getSavedSearches() {
    const data = await apiRequest('/searches');
    return data.searches;
}

async function deleteSearch(searchId) {
    return await apiRequest(`/searches/${searchId}`, {
        method: 'DELETE',
    });
}

async function bulkDeleteSearches(searchIds) {
    return await apiRequest('/searches/bulk-delete', {
        method: 'POST',
        body: JSON.stringify({ searchIds }),
    });
}

// OTP API functions
async function verifyOTP(email, otp) {
    const data = await apiRequest('/auth/verify-otp', {
        method: 'POST',
        body: JSON.stringify({ email, otp }),
    });

    setAuthToken(data.token);
    setCurrentUser(data.user);
    
    try {
        const subscription = await getSubscriptionStatus();
        if (!window.yopickySubscriptionCache) {
            window.yopickySubscriptionCache = {};
        }
        window.yopickySubscriptionCache[`sub_${data.user.id}`] = {
            data: subscription,
            timestamp: Date.now()
        };
    } catch (error) {
        // Silently ignore - subscription will be fetched when needed
    }
    
    return data;
}

async function resendOTP(email) {
    return await apiRequest('/auth/resend-otp', {
        method: 'POST',
        body: JSON.stringify({ email }),
    });
}

// Password reset API functions
async function forgotPassword(email) {
    try {
        const response = await fetch(`${API_BASE_URL}/auth/forgot-password`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ email }),
        });

        const data = await response.json();

        if (!response.ok) {
            return { success: false, error: data.error || 'Failed to send reset code' };
        }

        return { success: true, ...data };
    } catch (error) {
        console.error('Forgot password error:', error);
        return { success: false, error: 'Failed to connect to server' };
    }
}

async function resetPassword(email, code, newPassword) {
    try {
        const response = await fetch(`${API_BASE_URL}/auth/reset-password`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ email, code, newPassword }),
        });

        const data = await response.json();

        if (!response.ok) {
            return { success: false, error: data.error || 'Failed to reset password' };
        }

        return { success: true, ...data };
    } catch (error) {
        console.error('Reset password error:', error);
        return { success: false, error: 'Failed to connect to server' };
    }
}

async function changePassword(currentPassword, newPassword) {
    return await apiRequest('/auth/change-password', {
        method: 'POST',
        body: JSON.stringify({ currentPassword, newPassword }),
    });
}

// Collections API functions
async function createCollection(name, description = '') {
    return await apiRequest('/collections', {
        method: 'POST',
        body: JSON.stringify({ name, description }),
    });
}

async function getUserCollections() {
    const data = await apiRequest('/collections');
    return data.collections;
}

async function addPartToCollection(collectionId, partData) {
    return await apiRequest(`/collections/${collectionId}/parts`, {
        method: 'POST',
        body: JSON.stringify(partData),
    });
}

async function getCollectionParts(collectionId) {
    const data = await apiRequest(`/collections/${collectionId}/parts`);
    return data.parts;
}

async function deleteCollection(collectionId) {
    return await apiRequest(`/collections/${collectionId}`, {
        method: 'DELETE',
    });
}

async function removePartFromCollection(collectionId, partId) {
    return await apiRequest(`/collections/${collectionId}/parts/${partId}`, {
        method: 'DELETE',
    });
}

async function getPlans() {
    return await apiRequest('/subscription/plans');
}

async function getSubscriptionStatus() {
    return await apiRequest('/subscription/status');
}

async function createCheckoutSession(priceId) {
    return await apiRequest('/subscription/create-checkout', {
        method: 'POST',
        body: JSON.stringify({ priceId }),
    });
}

async function createPortalSession() {
    return await apiRequest('/subscription/create-portal', {
        method: 'POST',
    });
}

async function trackUsage() {
    return await apiRequest('/usage/track', {
        method: 'POST',
    });
}

async function getTodayUsage() {
    return await apiRequest('/usage/today');
}

async function searchParts(searchData) {
    return await apiRequest('/parts/search', {
        method: 'POST',
        body: JSON.stringify(searchData),
    });
}

async function getRepairCostEstimate(partData) {
    return await apiRequest('/ai/repair-cost', {
        method: 'POST',
        body: JSON.stringify(partData),
    });
}

async function getCarMarketValue(carData) {
    return await apiRequest('/car/market-value', {
        method: 'POST',
        body: JSON.stringify(carData),
    });
}

async function getCommunityParts(listingId, limit = 10) {
    return await apiRequest(`/parts/community/${listingId}?limit=${limit}`);
}

async function getPartSuggestions(damageDescription) {
    return await apiRequest(`/suggestions/damage?damageDescription=${encodeURIComponent(damageDescription)}`);
}

async function getCollectionCostEstimate(collectionId, includeBreakdowns = false) {
    return await apiRequest(`/collections/${collectionId}/estimate-cost`, {
        method: 'POST',
        body: JSON.stringify({ includeBreakdowns }),
    });
}

async function getCachedCollectionEstimate(collectionId) {
    return await apiRequest(`/collections/${collectionId}/estimate`);
}

const api = {
    register,
    login,
    logout,
    verifyToken,
    saveSearch,
    getSavedSearches,
    deleteSearch,
    bulkDeleteSearches,
    verifyOTP,
    resendOTP,
    forgotPassword,
    resetPassword,
    changePassword,
    createCollection,
    getUserCollections,
    addPartToCollection,
    getCollectionParts,
    deleteCollection,
    removePartFromCollection,
    getPlans,
    getSubscriptionStatus,
    createCheckoutSession,
    createPortalSession,
    trackUsage,
    getTodayUsage,
    searchParts,
    getRepairCostEstimate,
    getCarMarketValue,
    getCommunityParts,
    getPartSuggestions,
    getCollectionCostEstimate,
    getCachedCollectionEstimate,
};
